/**
 * @version 2.3.1
 * @change Fix for reload
 * incredibleSelects
 * @author Dan Slama
 **/


(function (global, factory) {
	if (typeof define === 'function' && define.amd) {
		// AMD. Register as an anonymous module.
		define(factory);
	} else if (typeof exports === 'object') {
		// CommonJS
		module.exports = factory();
	} else {
		// Browser globals
		global.incredibleSelects = factory();
	}
}(this, function () {
		'use strict';

		/**
		 * Polyfills and functions
		 * */

		(function () {
			if (typeof window.CustomEvent === 'function') return false;

			function CustomEvent(event, params) {
				params = params || {
					bubbles:    false,
					cancelable: false,
					detail:     undefined,
				};
				var evt = document.createEvent('CustomEvent');
				evt.initCustomEvent(event, params.bubbles, params.cancelable, params.detail);
				return evt;
			}

			CustomEvent.prototype = window.Event.prototype;

			window.CustomEvent = CustomEvent;
		})();

		function _extends() {
			_extends = Object.assign || function (target) {
				for (var i = 1; i < arguments.length; i++) {
					var source = arguments[i];

					for (var key in source) {
						if (Object.prototype.hasOwnProperty.call(source, key)) {
							target[key] = source[key];
						}
					}
				}

				return target;
			};
			return _extends.apply(this, arguments);
		}


		/**
		 * Initialization
		 * */

		var IncredibleSelects = function () {

			function IncredibleSelects(target, obj) {
				var o = typeof obj !== 'undefined' ? obj : {};

				var defaults = {
					closeOnSelect:       true,
					closeOnOutsideClick: true,
				};
				var translations = {
					'cs':    {
						'confirm': 'Potvrdit',
					},
					'en':    {
						'confirm': 'Confirm',
					},
					'other': {
						'confirm': 'Ok',
					},
				};
				var pluginLang = _lang;


				if (!translations[pluginLang]) {
					pluginLang = 'other';
				}


				this.userAgent = window.navigator.userAgent || 'no `userAgent` provided by the browser';
				this.props = _extends({}, defaults, o);
				this.translations = translations;
				this.pluginLang = pluginLang;

				this.instances = [];
				this.els = typeof target === 'string' ? document.querySelectorAll(target) : target;


				if (!('length' in this.els)) this.els = [this.els];
				for (var i = 0; i < this.els.length; i++) {
					var el = this.els[i].querySelector('.incredible-select__original-select')
						? this.els[i].querySelector('.incredible-select__original-select')
						: this.els[i];

					var instance = this.addInstance(el, this.props); // set vertical position
					this.instances.push(instance);
				}
			}

			var _proto = IncredibleSelects.prototype;

			_proto.addInstance = function (el, props) {
				var newProps = Object.assign({}, props);

				if (el.multiple) {
					newProps.closeOnSelect = false;
				}

				var item = {
					originalSelect:  el,
					parent:          el.parentNode,
					props:           newProps,
					stylableSelect:  null,
					stylableOptions: null,
					selectedOption:  null,
					canBeToggled:    true,
					placeholder:     '',
				};

				if (item.originalSelect.classList.contains('js--incredibly-initialized')) {
					this.reload(item);
				}

				this.render(item);
				return item;
			};

			_proto.render = function (item) {
				var _this = this;

				if (!item.originalSelect.classList.contains('js--incredibly-initialized')) {
					var stylableSelect = document.createElement('div');

					stylableSelect.className = item.originalSelect.className;
					stylableSelect.id = item.originalSelect.id;
					item.originalSelect.id = '';
					item.originalSelect.className = 'incredible-select__original-select js--incredibly-initialized';

					var incredibleObserver = new MutationObserver(function (mutations) {
						mutations.forEach(function (mutation) {
							if (mutation.type === 'attributes') {
								_proto.reload(item);
							}
						});
					});

					incredibleObserver.observe(item.originalSelect, {
						attributes:      true,
						attributeFilter: [
							'disabled',
							'class',
						],
						childList:       true,
						subtree:         true,
					});


					item.parent.insertBefore(stylableSelect, item.originalSelect);
					stylableSelect.appendChild(item.originalSelect);

					var selectedText = this.getSelected(item.originalSelect);
					item.originalSelect.style.display = 'none';

					stylableSelect.insertAdjacentHTML('afterbegin', '<div class=\'incredible-select__options\'></div>');
					stylableSelect.insertAdjacentHTML('afterbegin', '<div class=\'incredible-select__selected-option\'>' + selectedText + '</div>');

					this.getContainers(item);
					this.prepareOptions(item);

					item.stylableOptions.style.display = 'none';

					item.stylableOptions.addEventListener('click', function optionClick(e) {
						if (e.target.matches('.incredible-select__item:not(.is-disabled)')) {
							_this.optionOnClickListener(e, item);
						}
					});

					item.selectedOption.addEventListener('click', function selectClick(e) {
						if (!stylableSelect.classList.contains('is-disabled')) {
							_this.selectOnClickListener(e, item);
						}

						if (e.target.classList.contains('incredible-select__selected-cell') && item.originalSelect.multiple) {
							var option = item.stylableOptions.querySelector('.incredible-select__item[data-link-id=\'' + e.target.dataset.optionId + '\']');
							if (option) {
								option.click();
							}
						}
					});

					item.selectedOption.tabIndex = 0;

					item.selectedOption.onkeydown = function (e) {
						if (e.key === 'Enter' || e.key === ' ') {
							e.preventDefault();
							_proto.toggleOptions(item);
						}
						if (e.key === 'ArrowDown' || e.key === 'Down') {

							e.preventDefault();
							var firstOption = stylableSelect.querySelector('.incredible-select__item:first-of-type');
							firstOption.focus();
						}
					};

					document.addEventListener('click', function (e) {
						if (item.props.closeOnOutsideClick) {
							if (e.target === item.stylableOptions || item.stylableSelect.contains(e.target)) {
							} else {
								if (item.stylableOptions.style.display === 'block') {
									_this.closeOptions(item);
								}
							}
						}
					});


				}

			}

			_proto.getContainers = function (item) {

				item.stylableSelect = this.getClosestParent(item.originalSelect, '.incredible-select');

				if (item.originalSelect.hasAttribute('disabled')) {
					item.stylableSelect.classList.add('is-disabled');
				} else {
					item.stylableSelect.classList.remove('is-disabled');
				}

				item.stylableOptions = item.stylableSelect.querySelector('.incredible-select__options');
				item.selectedOption = item.stylableSelect.querySelector('.incredible-select__selected-option');
			};

			_proto.prepareOptions = function (item) {
				var selectedText = this.getSelected(item.originalSelect);

				item.stylableOptions.innerHTML = '';
				item.selectedOption.innerHTML = selectedText;

				let children = item.originalSelect.children;
				let iterations = 1;

				const findAndAppendOptions = function (children, appendToEl) {
					for (let i = 0; i < children.length; i++) {
						(function () {
							let child = children[i];
							if (child.matches('optgroup')) {
								const optGroup = document.createElement('div');
								optGroup.className = 'incredible-select__optgroup';

								const label = child.label || '';
								const groupLabel = document.createElement('DIV');
								groupLabel.innerHTML = label;
								groupLabel.className = 'incredible-select__label';
								optGroup.appendChild(groupLabel);

								findAndAppendOptions(child.children, optGroup);
								appendToEl.appendChild(optGroup);
							}
							if (child.matches('option')) {
								var currentOption = _proto.prepareOption(iterations, item, child);
								appendToEl.appendChild(currentOption);
							}

							iterations++;
						})();
					}
				};

				findAndAppendOptions(children, item.stylableOptions);
			};


			_proto.prepareOption = function (iteration, item, option) {

				var _this = this;

				option.dataset.linkId = iteration + '';
				var addClasses = '';

				if (option.selected) {
					addClasses += ' is-selected';
				}
				if (option.disabled) {
					addClasses += ' is-disabled';
				}

				var originalText = option.innerHTML;
				var el = document.createElement('div');
				var htmlString = '<div tabindex=\'0\' ' +
					'class=\'incredible-select__item ' +
					addClasses + '\' data-link-id=\'' +
					iteration + '\'>' + originalText + '</div>';

				el.innerHTML = htmlString;
				el = el.firstChild;

				if (item.originalSelect.multiple) {
					el.oncontextmenu = function (e) {
						e.preventDefault();
						e.stopPropagation();

						var bar = document.createElement('div');
						bar.classList.add('incredible-select__bar');

						var barBtn = document.createElement('div');
						barBtn.classList.add('incredible-select__close-btn');
						barBtn.innerHTML =
							_this.translations
								[_this.pluginLang]
								['confirm'];

						barBtn.onclick = function () {
							item.canBeToggled = true;
							_proto.closeOptions(item);
							bar.remove();
						};

						bar.appendChild(barBtn);
						item.stylableOptions.appendChild(bar);
						item.canBeToggled = false;
						_proto.setActiveOption(e, item, false);
					};
				}

				el.onkeydown = function (e) {
					if (e.key === 'Enter' || e.key === ' ') {
						e.preventDefault();
						_proto.setActiveOption(e, item);
						item.selectedOption.focus();
					}

					if (e.key === 'ArrowUp' || e.key === 'Up') {
						e.preventDefault();
						if (el.previousElementSibling) {
							el.previousElementSibling.focus();
						} else {
							item.selectedOption.focus();
						}
					}

					if (e.key === 'ArrowDown' || e.key === 'Down') {
						e.preventDefault();
						if (el.nextElementSibling) {
							el.nextElementSibling.focus();
						}
					}
				};

				return el;
			};

			_proto.selectOnClickListener = function (event, item) {
				_proto.toggleOptions(item);
			};

			_proto.optionOnClickListener = function (event, item) {
				_proto.setActiveOption(event, item);
			};

			_proto.setActiveOption = function (event, item, keepOpened = false) {
				var option = item.originalSelect.querySelector('option[data-link-id=\'' + event.target.dataset.linkId + '\']');
				option.selected = !option.selected;

				item.selectedOption.innerHTML = this.getSelected(item.originalSelect);

				if (!item.originalSelect.multiple) {
					var stylableOptionsItems = item.stylableOptions.querySelectorAll('.incredible-select__item');
					for (var i = 0; i < stylableOptionsItems.length; i++) {
						stylableOptionsItems[i].classList.remove('is-selected');
					}
				}

				var customEvent = new CustomEvent('change');
				item.originalSelect.dispatchEvent(customEvent);

				if (item.props.closeOnSelect && !keepOpened) {
					item.stylableSelect.classList.toggle('is-expanded');
					_proto.toggleOptions(item);
				}

				event.target.classList.toggle('is-selected');
			};


			_proto.toggleOptions = function (item) {
				if (item.canBeToggled) {
					this.toggle(item.stylableOptions);
					item.stylableSelect.classList.toggle('is-expanded');
				}
			};

			_proto.closeOptions = function (item) {
				if (item.canBeToggled) {
					item.stylableOptions.style.display = 'none';
					item.stylableSelect.classList.remove('is-expanded');
				}
			};

			_proto.getClosestParent = function (el, selector) {
				for (; el && el !== document; el = el.parentNode) {
					if (el.matches(selector)) return el;
				}
				return null;
			};

			_proto.reload = function (item) {
				this.getContainers(item);
				item.stylableOptions.innerHTML = '';
				this.prepareOptions(item);
			};

			_proto.getSelected = function (el) {
				var selectedText = '';
				var checkedOptions = el.querySelectorAll('option:checked');

				for (var i = 0; i < checkedOptions.length; ++i) {
					selectedText += (checkedOptions[i].innerHTML) + ', ';
				}

				if (selectedText === '' && el.dataset.placeholder) {
					return '<span class="incredible-select__placeholder">'
						+ el.dataset.placeholder
						+ '</span>';
				} else {
					return selectedText.slice(0, -2);
				}
			};

			_proto.toggle = function (el) {

				if (window.getComputedStyle(el).display === 'none') {
					el.style.display = 'block';
				} else {
					el.style.display = 'none';
				}
			};


			return IncredibleSelects;
		}();

		function incredibleSelects(target, o) {
			return new IncredibleSelects(target, o);
		}

		return incredibleSelects;
	}
));